(function() {
  var sender_domain = document.location.ancestorOrigins[0]; // Get domain
  // Regular expression to match the domains
  var domainRegex = /^https?:\/\/([a-z0-9]+[.])*aaardvarkaccessibility\.[a-z]{2,4}$/;
  
  if (typeof sender_domain === 'undefined' || !sender_domain.match(domainRegex)) {
    return;
  }

  window.onload = function () {
    // If the sender domain is one of the valid domains
    createStyleElement();
    handleUrlChange();
    
  };
    
  var isHoverEffectEnabled = true; // Flag to track the hover effect status
  var handel_input;

  function stopAction(e) {
    e.preventDefault();
  }

  function getSelector(elm) {
    if (elm.tagName === "BODY") return "BODY";
    const names = [];
    while (elm.parentElement && elm.tagName !== "BODY") {
      if (elm.id) {
        names.unshift("#" + elm.getAttribute("id")); // getAttribute, because `elm.id` could also return a child element with name "id"
        break; // Because ID should be unique, no more is needed. Remove the break, if you always want a full path.
      } else {
        let c = 1,
          e = elm;
        for (; e.previousElementSibling; e = e.previousElementSibling, c++);
        names.unshift(elm.tagName + ":nth-child(" + c + ")");
      }
      elm = elm.parentElement;
    }
    return names.join(">");
  }

  function sendElement(event) {
    let obj_boundaries = event.target.getBoundingClientRect();
    let output = getSelector(event.target);
    let element = {
      response: "annotate",
      coordinates: obj_boundaries,
      selector: output,
    };
    window.parent.postMessage(element, sender_domain);

    isHoverEffectEnabled = false;
    removeHoverEffectFromElements();
    removeHoverEffectListeners();
  }

  window.addEventListener(
    "message",
    (handel_input = (event) => {
      if (event.origin !== sender_domain) {
        return; // Skip message domain is not trusted
      }

	  if(event.data.function == 'handShake'){
			window.parent.postMessage(
            {
              msg: "ACK"
            },
            sender_domain
          );
		  handleUrlChange();
		  createStyleElement();
	  }
      else if (event.data.function == "annotate") {
        isHoverEffectEnabled = true;
        applyHoverEffect();
        let links = document.getElementsByTagName("a");
        let array = [...links];
        array.forEach((link) => {
          link.addEventListener("click", stopAction);
        });
        document.body.addEventListener("click", sendElement);
      } else if (event.data.function == "getElementBoundaries") {
        let element = document.querySelector(event.data.selector);
        if (element != null) {
          let element_boundaries = element.getBoundingClientRect();
          window.parent.postMessage(
            {
              msg: "getElementBoundaries",
              element_boundaries: element_boundaries,
            },
            sender_domain
          );
        } else {
          window.parent.postMessage(
            { msg: "getElementBoundaries", element_boundaries: null },
            sender_domain
          );
        }
      } else if (event.data.function == "defaultBehavior") {
        document.body.removeEventListener("click", sendElement);
        let links = document.getElementsByTagName("a");
        let array = [...links];
        array.forEach((link) => {
          link.removeEventListener("click", stopAction);
        });
        window.parent.postMessage("restored default behavior!", sender_domain);
      } else if (event.data.function == "init") {
        var height = getIframeHeight();
        window.parent.postMessage({ iframe_height: height }, sender_domain);
      } else if (event.data.function == "resize") {
        window.addEventListener("resize", myEfficientFn);
      } else if (event.data.function == "remove") {
        removeListeners();
      } else if (event.data.function == "listen_for_hidden") {
        listenForHidden();
      } else if (event.data.function == "iframe_mutation") {
        observeLazyLoading();
      } else if (event.data.function == "setVhInPx") {
        setVhInPx(event.data.vhInPx);
      }
    })
  );

  let globalVhInPx;
  async function setVhInPx(vhInPx) {
    if(!vhInPx || vhInPx === globalVhInPx) {
      return;
    };
    globalVhInPx = vhInPx;
    await modifyAndInjectStylesheets();
    modifyInlineVhStyle();
    postNewHeight();
  }

  function debounce(func, wait, immediate) {
    var timeout;
    return function () {
      var context = this,
        args = arguments;
      var later = function () {
        timeout = null;
        if (!immediate) func.apply(context, args);
      };
      var callNow = immediate && !timeout;
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
      if (callNow) func.apply(context, args);
    };
  }

  var myEfficientFn = debounce(function () {
    window.parent.postMessage("reload", sender_domain);
  }, 250);

  function removeListeners() {
    window.removeEventListener("message", myEfficientFn);
    window.removeEventListener("message", sendElement);
    window.removeEventListener("message", stopAction);
    observer.disconnect();
    // window.removeEventListener('message', handel_input)
  }

// Setup a MutationObserver to listen for nodes being added or attributes being modified
let lazyLoadingObserver;
function observeLazyLoading() {
  lazyLoadingObserver = new MutationObserver((mutations) => {
    let shouldPostNewHeight = false;
    for(let mutation of mutations) {
        if(isRelevantToHeight(mutation)) {
            shouldPostNewHeight = true;
        }
        if(isRelevantToVhUnit(mutation)) {
          handleMutationInlineVhStyle(mutation);
        }
    }
    if(shouldPostNewHeight){
      postNewHeight();
    }
  });
  
  let config = {
      attributes: true, 
      childList: true, 
      subtree: true
  };
  
  lazyLoadingObserver.observe(document.body, config);
}

function isRelevantToHeight(mutation) {
  return mutation.type === 'childList' && (mutation.addedNodes.length > 0 || mutation.removedNodes.length > 0);
}
function isRelevantToVhUnit(mutation) {
  return mutation.type === 'childList' && mutation.addedNodes.length > 0;
}

function getIframeHeight() {
  let bodyHeight = document.body.scrollHeight;
  let htmlHeight = document.documentElement.scrollHeight;
  return Math.max(bodyHeight, htmlHeight);
}


// Function to calculate the new height and post a message to the parent window
let lastSentHeight = 0;
let resizeTimeout;
function postNewHeight() {
  clearTimeout(resizeTimeout); // Clear any existing timeout
  resizeTimeout = setTimeout(() => {
    let newHeight = getIframeHeight();
    if(Math.abs(newHeight - lastSentHeight) > 1) {
        window.parent.postMessage({ msg: "updateHeight", iframe_height: newHeight }, sender_domain);
        lastSentHeight = newHeight;
      }
    }, 200);
}

  let observer;
  function listenForHidden() {
    let options = {
      attributes: true,
      characterData: true,
      childList: true,
      subtree: true,
      attributeOldValue: true,
      characterDataOldValue: true,
    };
    observer = new MutationObserver(mCallback);
    observer.observe(document.documentElement, options);
  }

  function mCallback(mutations) {
    for (let mutation of mutations) {
      // console.log(`Mutation Detected: Mutation name is:`,mutation)
      window.parent.postMessage("reload_hidden", sender_domain);
      // observer.disconnect()
    }
  }
  function createStyleElement() {
    let style = document.createElement("style");
    style.textContent = `
          .hover-outline {
              outline: 2px solid #D2042D !important;
              position: relative;
          }

          .hover-overlay {
              position: absolute;
              z-index: 9999;
              background-color: #D2042D;
              padding: 1px;
              font-weight: bold;
              font-size: 12px;
              top: -20px;
              left: -20px;
              color: white;
          }

          .hover-overlay-text {
              display: inline-block;
              padding: 5px;
              background-color: #D2042D;
          }
      `;
    document.head.appendChild(style);
  }
  function applyHoverEffect() {
    let elements = document.querySelectorAll("*");
    elements.forEach(function (element) {
      element.addEventListener("mouseover", addHoverEffect);
      element.addEventListener("mouseout", removeHoverEffect);
    });
  }

  function addHoverEffect(event) {
    if (!isHoverEffectEnabled) {
      return; // Exit if the hover effect is disabled
    }
    event.stopPropagation();
    event.target.classList.add("hover-outline");

    // Create the text overlay
    let overlay = document.createElement("div");
    overlay.classList.add("hover-overlay");

    let selector = getSelector(event.target);
    let shortSelector = getShortSelector(selector);

    let text = document.createElement("div");
    text.classList.add("hover-overlay-text");
    text.textContent = shortSelector;

    overlay.appendChild(text);

    // Position the text overlay relative to the viewport
    let elementRect = event.target.getBoundingClientRect();
    let scrollLeft = window.pageXOffset || document.documentElement.scrollLeft;
    let scrollTop = window.pageYOffset || document.documentElement.scrollTop;

    overlay.style.left = elementRect.left - 2 + scrollLeft + "px";
    document.body.appendChild(overlay);
    if (elementRect.top <= overlay.clientHeight) {
      overlay.style.top = elementRect.top + scrollTop + "px";
    } else {
      overlay.style.top = elementRect.top - overlay.clientHeight + scrollTop + "px";
    }
  }

  function getShortSelector(selector) {
    // Trim the selector and split it into parts
    let parts = selector.trim().split(">");

    // Get the first part and the last part of the selector
    let firstPart = parts[0].trim().toLowerCase();
    let lastPart = parts[parts.length - 1].trim().toLowerCase();

    // Get the element by the given selector
    let element = document.querySelector(selector);

    // Get the tag/type of the element
    let elementType = element ? element.tagName.toLowerCase() : "";

    // Construct the short selector with tag/type and ID (if present)
    let shortSelector = elementType;
    let idMatch = firstPart.match(/#([^.\s]+)/);
    if (idMatch) {
      let id = "#" + idMatch[1];
      shortSelector += " " + id;
    } else {
      shortSelector += " " + firstPart;
    }

    // add the last part if it's not the same as the first part
    if (shortSelector.length > 0 && firstPart != lastPart) {
      shortSelector += "..." + lastPart;
    }

    return shortSelector;
  }

  function removeHoverEffect(event) {
    if (!isHoverEffectEnabled) {
      return; // Exit if the hover effect is disabled
    }
    event.stopPropagation();
    event.target.classList.remove("hover-outline");

    // Remove the text overlay
    let overlays = document.querySelectorAll(".hover-overlay");
    overlays.forEach(function (overlay) {
      overlay.remove();
    });
  }
  function removeHoverEffectFromElements() {
    let elements = document.querySelectorAll(".hover-outline");
    elements.forEach(function (element) {
      element.classList.remove("hover-outline");
    });

    let overlays = document.querySelectorAll(".hover-overlay");
    overlays.forEach(function (overlay) {
      overlay.remove();
    });
  }
  function removeHoverEffectListeners() {
    let elements = document.querySelectorAll("*");
    elements.forEach(function (element) {
      element.removeEventListener("mouseover", addHoverEffect);
      element.removeEventListener("mouseout", removeHoverEffect);
    });
  }

  // Define a function to handle URL changes
  function handleUrlChange(event) {
		const urlWithoutTrailingSlash = window.location.pathname.replace(/\/$/, '');
		const urlToSend = urlWithoutTrailingSlash === '' ? '/' : urlWithoutTrailingSlash;
		window.parent.postMessage(
			{ msg: "urlChanged", url: urlToSend },
			sender_domain
		);
	}
  
  /* Mofify style sheets for VH unit */
  let originalStyleSheets = null;
  async function modifyAndInjectStylesheets() {
    if(!globalVhInPx) return;
    VhInPx = globalVhInPx;

    // Store original stylesheets if not already stored
    if (!originalStyleSheets) {
        originalStyleSheets = [];
        for (const sheet of document.styleSheets) {
          let cssText = null;
          if (sheet.href) {
              const cssText = await fetch(sheet.href).then(response => response.text()).catch(error => console.error('Error fetching the stylesheet', error));
          } else if (sheet.ownerNode.tagName.toLowerCase() === 'style') {
            cssText = sheet.ownerNode.textContent;
          }
          if (cssText && cssText.includes('vh')) {
            originalStyleSheets.push({ cssText });
          }
        }
    }

    if (originalStyleSheets.length === 0) return;

    // Check and remove existing modified stylesheet
    const existingModifiedStylesheet = document.getElementById('modified-stylesheet');
    if (existingModifiedStylesheet) {
        existingModifiedStylesheet.parentNode.removeChild(existingModifiedStylesheet);
    }

    const newStyleElement = document.createElement('style');
    newStyleElement.id = 'modified-stylesheet'; // Assign an id to the new stylesheet

    let contentArray = [];
    originalStyleSheets.forEach(sheet => {
        let cssText = sheet.cssText;
        cssText = cssText.replace(/([\d.]+)vh\b/g, (match, p1) => `${((p1 * VhInPx) / 100).toFixed(2)}px !important`);
        contentArray.push(cssText);
    });
    newStyleElement.textContent = contentArray.join("");
    document.head.prepend(newStyleElement);
}
/* End Mofify style sheets for VH unit */

/* Inline VH unit style handling */
function modifyInlineVhStyle(){
  resetVhInlineStyles();
  modifyVhInlineStyles();
}
let elementsWithVhInlineStyles = new Map();
function resetVhInlineStyles() {
  elementsWithVhInlineStyles.forEach((originalStyle, element) => {
      element.setAttribute('style', originalStyle);
  });
}
function modifyVhInlineStyles() {
    // If this is the first run, identify and store elements with vh in inline styles
    if (elementsWithVhInlineStyles.size === 0) {
        const allElements = document.getElementsByTagName("*");
        for (const element of allElements) {
            if (element.hasAttribute('style')) {
                let style = element.getAttribute('style');
                if (/[\d.]+vh\b/.test(style)) {
                    elementsWithVhInlineStyles.set(element, style);
                }
            }
        }
    }
    
    // Replace vh units in the stored elements with their px equivalent
    elementsWithVhInlineStyles.forEach((originalStyle, element) => {
        let modifiedStyle = originalStyle.replace(/([\d.]+)vh\b/g, (match, p1) => `${((p1 * globalVhInPx) / 100).toFixed(2)}px`);
        element.setAttribute('style', modifiedStyle);
    });
}
function handleMutationInlineVhStyle(mutation){
  if(!globalVhInPx){
    return;
  }
  mutation.addedNodes.forEach((node) => {
    if (node.nodeType === Node.ELEMENT_NODE && node.hasAttribute('style')) {
        let style = node.getAttribute('style');
        if (/[\d.]+vh\b/.test(style)) {
            elementsWithVhInlineStyles.set(node, style);
            let modifiedStyle = style.replace(/([\d.]+)vh\b/g, (match, p1) => `${((p1 * globalVhInPx) / 100).toFixed(2)}px`);
            node.setAttribute('style', modifiedStyle);
        }
    }
  });
}
/* End Inline VH unit style handling */


})();